<?php
/**
 * Elementor Widget
 * @package bizmaster
 * @since 1.0.0
 */

namespace Elementor;

class Bizmaster_Blog_Slider_One_Widget extends Widget_Base
{
    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'bizmaster-blog-slider-one-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Blog Slider One', 'bizmaster-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-posts-carousel';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['bizmaster_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
		$this->start_controls_section(
            'slider_settings_section',
			[
				'label' => esc_html__('Slider Settings', 'bizmaster-core'),
				'tab' => Controls_Manager::TAB_CONTENT
			]
		);

		$this->add_control('total', [
            'label' => esc_html__('Total Posts', 'bizmaster-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many post you want in slider , enter -1 for unlimited post.')
        ]);

        $this->add_control('category', [
            'label' => esc_html__('Category', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => bizmaster()->get_terms_names('category', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'bizmaster-core'),
        ]);

		$this->add_control('order', [
            'label' => esc_html__('Order', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'bizmaster-core'),
                'DESC' => esc_html__('Descending', 'bizmaster-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'bizmaster-core')
        ]);

        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'bizmaster-core'),
                'title' => esc_html__('Title', 'bizmaster-core'),
                'date' => esc_html__('Date', 'bizmaster-core'),
                'rand' => esc_html__('Random', 'bizmaster-core'),
                'comment_count' => esc_html__('Most Comments', 'bizmaster-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'bizmaster-core')
        ]);

		$this->add_control(
            'items',
            [
                'label' => esc_html__('Slides To Show', 'bizmaster-core'),
                'type' => Controls_Manager::NUMBER,
                'description' => esc_html__('you can set how many item show in slider', 'bizmaster-core'),
                'default' => '3',
            ]
        );

		$this->add_control(
            'dots',
            [
                'label' => esc_html__('Dots', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Loop', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

		$this->add_control(
            'center',
            [
                'label' => esc_html__('Center', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),

            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

		$this->add_control(
            'autoplaytimeout',
            [
                'label' => esc_html__('Autoplay Timeout', 'bizmaster-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 500,
                        'max' => 10000,
                        'step' => 100,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5000,
                ],
                'size_units' => ['px'],
                'condition' => array(
                    'autoplay' => 'yes'
                )
            ]
        );

		$this->end_controls_section();

		$this->start_controls_section(
			'blog_styling_settings_section',
			[
				'label' => esc_html__('Styling Settings', 'bizmaster-core'),
				'tab' => Controls_Manager::TAB_STYLE
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => esc_html__('Image Border Radius', 'bizmaster-core'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .blog-card .blog-img img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control('blog_content_bg_color', [
			'label' => esc_html__('Blog Content BG Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card .blog-content" => "background: {{VALUE}}"
			]
		]);

		$this->add_control('blog_content_bg_hover_color', [
			'label' => esc_html__('Blog Content BG Hover Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card:hover .blog-content" => "background: {{VALUE}}"
			]
		]);

		$this->add_control(
			'blog_content_border_radius',
			[
				'label'      => esc_html__('Blog Content Border Radius', 'bizmaster-core'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .blog-card .blog-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control('title_color', [
			'label' => esc_html__('Title Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card .blog-content .blog-title a" => "color: {{VALUE}}"
			]
		]);

		$this->add_control('title_hover_color', [
			'label' => esc_html__('Title Hover Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card .blog-content .blog-title a:hover" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title Typography', 'bizmaster-core'),
            'name' => 'title_typography',
            'description' => esc_html__('Title Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .blog-card .blog-content .blog-title a"
        ]);

		$this->add_control('blog_icon_color', [
			'label' => esc_html__('Blog Meta Icon Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card .blog-content .blog-meta a svg" => "color: {{VALUE}}; fill: {{VALUE}}",
				"{{WRAPPER}} .blog-card .blog-content .blog-meta a i" => "color: {{VALUE}}"
			]
		]);

		$this->add_control('blog_icon_hover_color', [
			'label' => esc_html__('Blog Meta Icon Hover Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card .blog-content .blog-meta a:hover svg" => "color: {{VALUE}}; fill: {{VALUE}}",
				"{{WRAPPER}} .blog-card .blog-content .blog-meta a:hover i" => "color: {{VALUE}}"
			]
		]);

		$this->add_control('blog_meta_color', [
			'label' => esc_html__('Blog Meta Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card .blog-content .blog-meta a" => "color: {{VALUE}}"
			]
		]);

		$this->add_control('blog_meta_hover_color', [
			'label' => esc_html__('Blog Meta Hover Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .blog-card .blog-content .blog-meta a:hover" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Blog Meta Typography', 'bizmaster-core'),
            'name' => 'blog_meta_typography',
            'description' => esc_html__('Blog Meta Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .blog-card .blog-content .blog-meta p, {{WRAPPER}} .blog-card .blog-content .blog-meta a"
        ]);

		$this->add_control('slick_dots_bg_color', [
            'label' => esc_html__('Slick Dots BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
			'selectors' => [
                "{{WRAPPER}} .slick-dots button" => "background-color: {{VALUE}}",
            ]
        ]);

        $this->add_control('slick_active_dot_bg_color', [
            'label' => esc_html__('Slick Active Dot BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .slick-dots .slick-active button" => "background-color: {{VALUE}}",
            ]
        ]);

		$this->add_control('slick_active_dot_border_color', [
            'label' => esc_html__('Slick Active Dot Border Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .slick-dots .slick-active button::before" => "border-color: {{VALUE}}",
            ]
        ]);

		$this->end_controls_section();
	}

    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
	{
		$settings = $this->get_settings_for_display();
		$rand_numb = rand(333, 999999999);

		//slider settings
        $slider_settings = [
			"dots" => esc_attr($settings['dots']),
			"loop" => esc_attr($settings['loop']),
			"items" => esc_attr($settings['items'] ? $settings['items'] : 1),
            "center" => esc_attr($settings['center']),
            "autoplay" => esc_attr($settings['autoplay']),
            "autoplaytimeout" => esc_attr(!empty($settings['autoplaytimeout']['size']) ? $settings['autoplaytimeout']['size'] : 0)
		];

		//query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];

		//setup query
        $args = array(
            'post_type' => 'post',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'category',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
		?>
		<div class="row blog-slider slider-shadow" id="blog-carousel-one-<?php echo esc_attr($rand_numb); ?>" data-settings='<?php echo json_encode($slider_settings); ?>'>
		<?php
			while ($post_data->have_posts()) : $post_data->the_post();

				//image condition here
				$img_id = get_post_thumbnail_id(get_the_ID()) ? get_post_thumbnail_id(get_the_ID()) : false;
				$img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'full', false) : '';
				$img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
				$img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
				$comments_count = get_comments_number(get_the_ID());
				$comment_text = ($comments_count > 1) ? 'Comments (' . $comments_count . ')' : 'Comment (' . $comments_count . ')';
			?>
				<div class="col-md-6 col-lg-4">
					<div class="blog-card">
						<div class="blog-img">
							<img src="<?php echo esc_url($img_url); ?>" alt="<?php echo $img_alt; ?>">
						</div>
						<div class="blog-content">
							<div class="blog-meta">
								<p><a href="<?php echo get_author_posts_url(get_the_author_meta('ID')); ?>"><i class="fas fa-user"></i><?php the_author(); ?></a></p>
								<p><a href="<?php comments_link(); ?>"><i class="fas fa-comments"></i><?php echo $comment_text; ?></a></p>
							</div>
							<h3 class="blog-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
						</div>
					</div>
				</div>
			<?php endwhile; ?>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type(new Bizmaster_Blog_Slider_One_Widget());
