<?php
/**
 * Elementor Widget
 * @package bizmaster
 * @since 1.0.0
 */

namespace Elementor;

class Bizmaster_Team_Slider_Two_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'bizmaster-team-slider-two-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Team Slider Two', 'bizmaster-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['bizmaster_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
		$this->start_controls_section(
            'slider_settings_section',
            [
                'label' => esc_html__('Slider Settings', 'bizmaster-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

		$this->add_control('total', [
            'label' => esc_html__('Total Posts', 'bizmaster-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many post you want in slider, enter -1 for unlimited post.')
        ]);
		
		$this->add_control('category', [
            'label' => esc_html__('Category', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => bizmaster()->get_terms_names('team-category', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'bizmaster-core'),
        ]);

        $this->add_control('order', [
            'label' => esc_html__('Order', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'bizmaster-core'),
                'DESC' => esc_html__('Descending', 'bizmaster-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'bizmaster-core')
        ]);

        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'bizmaster-core'),
                'title' => esc_html__('Title', 'bizmaster-core'),
                'date' => esc_html__('Date', 'bizmaster-core'),
                'rand' => esc_html__('Random', 'bizmaster-core'),
                'comment_count' => esc_html__('Most Comments', 'bizmaster-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'bizmaster-core')
        ]);

		$this->add_control(
            'items',
            [
                'label' => esc_html__('Slides To Show', 'bizmaster-core'),
                'type' => Controls_Manager::NUMBER,
                'description' => esc_html__('you can set how many item show in slider', 'bizmaster-core'),
                'default' => '3',
            ]
        );

		$this->add_control(
            'dots',
            [
                'label' => esc_html__('Dots', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Loop', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

		$this->add_control(
            'center',
            [
                'label' => esc_html__('Center', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),

            ]
        );

		$this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

		$this->add_control(
            'autoplaytimeout',
            [
                'label' => esc_html__('Autoplay Timeout', 'bizmaster-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 500,
                        'max' => 10000,
                        'step' => 100,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5000,
                ],
                'size_units' => ['px'],
                'condition' => array(
                    'autoplay' => 'yes'
                )
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
            'team_member_styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'bizmaster-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_control('share_icons_bg_color', [
            'label' => esc_html__('Share Icons BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-box .team-social_wrap .social-btn.style2 a" => "background-color: {{VALUE}}"
            ]
        ]);

        $this->add_control('share_icon_color', [
            'label' => esc_html__('Share Icon Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-box .team-social_wrap .social-btn.style2 a" => "color: {{VALUE}}",
            ]
        ]);

		$this->add_control('share_icons_bg_hover_color', [
            'label' => esc_html__('Share Icons BG Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-box .team-social_wrap .social-btn.style2 a:hover" => "background-color: {{VALUE}}"
            ]
        ]);

        $this->add_control('share_icon_hover_color', [
            'label' => esc_html__('Share Icon Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-box .team-social_wrap .social-btn.style2 a:hover" => "color: {{VALUE}}",
            ]
        ]);

		$this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-box_content h4.team-box_title a" => "color: {{VALUE}}",
            ]
        ]);

        $this->add_control('title_hover_color', [
            'label' => esc_html__('Title Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-box_content h4.team-box_title a:hover" => "color: {{VALUE}}",
            ]
        ]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title Typography', 'bizmaster-core'),
            'name' => 'title_typography',
            'description' => esc_html__('Title Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .team-box_content h4.team-box_title a"
        ]);

		$this->add_control('subtitle_color', [
            'label' => esc_html__('Subtitle Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-box_content .team-box_desig" => "color: {{VALUE}}",
            ]
        ]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Subtitle Typography', 'bizmaster-core'),
            'name' => 'subtitle_typography',
            'description' => esc_html__('Subtitle Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .team-box_content .team-box_desig"
        ]);

		$this->add_control('slick_dots_bg_color', [
            'label' => esc_html__('Slick Dots BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
			'default' => '#54595F',
			'selectors' => [
                "{{WRAPPER}} .slick-dots" => "background-color: {{VALUE}}",
            ]
        ]);

        $this->add_control('slick_active_dot_bg_color', [
            'label' => esc_html__('Slick Active Dot BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .slick-dots .slick-active button" => "background-color: {{VALUE}}",
            ]
        ]);

        $this->end_controls_section();
	}

    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
		$settings = $this->get_settings_for_display();
		$rand_numb = rand(333, 999999999);

		//slider settings
		$slider_settings = [
			"dots" => esc_attr($settings['dots']),
			"loop" => esc_attr($settings['loop']),
            "items" => esc_attr($settings['items'] ? $settings['items'] : 1),
            "center" => esc_attr($settings['center']),
            "autoplay" => esc_attr($settings['autoplay']),
			"autoplaytimeout" => esc_attr(!empty($settings['autoplaytimeout']['size']) ? $settings['autoplaytimeout']['size'] : 0)
		];

        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];

        //setup query
        $args = array(
            'post_type' => 'team',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'team-category',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
		?>
		<div id="team-two-carousel-<?php echo esc_attr($rand_numb); ?>" data-settings='<?php echo json_encode($slider_settings) ?>' class="row gx-30 team-slider2">
			<?php
			while ($post_data->have_posts()) : $post_data->the_post();
				$post_id = get_the_ID();
				$img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
				$img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'full', false) : '';
				$img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
				$img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
				$team_single_meta_data = get_post_meta(get_the_ID(), 'bizmaster_team_options', true);
				$social_icons = isset($team_single_meta_data['social-icons']) && !empty($team_single_meta_data['social-icons']) ? $team_single_meta_data['social-icons'] : '';
			?>

				<div class="team-box m-3">
					<div class="team-box_img">
						<img src="<?php echo esc_url($img_url) ?>" alt="<?php echo $img_alt; ?>">
					</div>
					<div class="team-box_content">
						<h4 class="team-box_title"><a href="<?php the_permalink() ?>"><?php echo esc_html(get_the_title($post_id)) ?></a></h4>
						<span class="team-box_desig"><?php echo $team_single_meta_data['designation'] ?></span>
						<div class="team-social_wrap">
							<div class="social-btn style2">
								<?php
									if (!empty($social_icons)) {
										foreach ($social_icons as $item) {
											printf('<a href="%1$s"><i class="%2$s"></i></a>', $item['url'], $item['icon']);
										}
									}
								?>
							</div>
						</div>
					</div>
				</div>

			<?php endwhile; ?>
		</div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Bizmaster_Team_Slider_Two_Widget());
